//
//  $Id: WXKPhone.m 108 2009-06-24 14:54:10Z fujidana $
//  Copyright 2006 FUJIDANA. All rights reserved.
//
//
//  Some part of this source code is derived from AirHPhone.m, 
//  written by raktajino in AH-K3001V Bookmark Utility and distributed
//  under BSD license.
//  Communicational protocol to AH-K3001V was referred to ``AirH" 
//  AH-K3001 station'' (http://www.softclub.jp/zoro/ce/usb.html), written
//  by Zoroyoshi and distributed under BSD license.
//  See the following about copyrights.
//
//
//  Created by raktajino on Thu Jun 17 2004.
//  Copyright (c) 2004 raktajino. All rights reserved.
//
//		Copyright (c) 04 Zoroyoshi, Japan
//		All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "WXKPhone.h"
#import "WXKError.h"
#import "WXKProgressing.h"

#import <stdio.h>
#import <unistd.h>
#import <errno.h>
#import <fcntl.h>
#import <sys/ioctl.h>

#define DEBUG 1


typedef enum
{
	WXKPhoneGeneralOperation = 1,
	WXKPhoneReceiveItemsOperation = 2,
	WXKPhoneSendItemOperation = 3,
	WXKPhoneDeleteOperation = 4,
	WXKPhoneBeginSendAllOperation = 5,
	WXKPhoneFinishSendAllOperation = 6,
	WXKPhoneReceiveDirectoryOperation = 7
} WXKPhoneOperation;

static const unsigned kOpeningRetryCount = 4;             // max retry number on open/log-in.
static const NSTimeInterval kOpeningRetryInterval = 0.25; // retry interval on open/log-in. (sec)
static const unsigned kRequireDriverMajorVersion = 1;     // required major verision number of AH-K3001V USB Driver
static const unsigned kRequireDriverMinorVersion = 2;     // required minor verision number of AH-K3001V USB Driver
static const unsigned kReadBufferSize = 4096;             // preferred buffer size for reading


@interface WXKPhone ()

- (NSArray *)receivedDataArrayOfType:(WXKPhoneDataType)dataType operation:(WXKPhoneOperation)operation error:(NSError **)errorPtr;
- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(char *)parameter error:(NSError **)errorPtr;
- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b data:(NSData *)data error:(NSError **)errorPtr;
- (NSData *)receivedDataWithAddition:(BOOL)hasAdditionalData error:(NSError **)errorPtr;
- (BOOL)sendRawBytes:(const UInt8 *)buffer length:(SInt32)length error:(NSError **)errorPtr;
- (SInt32)receiveRawBytes:(UInt8 *)buffer maxLength:(SInt32)bufferSize error:(NSError **)errorPtr;

@end


@implementation WXKPhone

#pragma mark Public class methods

+ (BOOL)existsDriver
{
	NSArray *paths = NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, NSSystemDomainMask, YES);
	if (paths == nil || [paths count] == 0) return NO;
	
	NSString *driverPath = [[[paths lastObject] stringByAppendingPathComponent:@"Extensions"] stringByAppendingPathComponent:@"AHK3001VDriver.kext"];
	NSBundle *driverBundle = [NSBundle bundleWithPath:driverPath];
	if (driverBundle == nil) return NO;
	
	NSString *versionString = [driverBundle objectForInfoDictionaryKey:@"CFBundleShortVersionString"];
	if (versionString == nil) return NO;
	
	NSArray *versionArray = [versionString componentsSeparatedByString:@"."];
	if ([versionArray count] < 2) return NO;
	
	SInt32 currentMajorVersion = [[versionArray objectAtIndex:0] intValue];
	SInt32 currentMinorVersion = [[versionArray objectAtIndex:1] intValue];
	
	if (currentMajorVersion > kRequireDriverMajorVersion) return YES;
	else if (currentMajorVersion == kRequireDriverMajorVersion && currentMinorVersion >= kRequireDriverMinorVersion) return YES;
	
	return NO;
}

+ (NSString *)requiredDriverName
{
	return [NSString stringWithFormat:@"AH-K3001V USB Driver %d.%d", kRequireDriverMajorVersion, kRequireDriverMinorVersion];
}

+ (BOOL)getParsedDataFields:(NSData **)dataFields fieldNumber:(unsigned)fieldNum fromData:(NSData *)rawData
{
	const UInt8 *bytes = [rawData bytes];
	SInt32 readLength = 13;
	
	while (readLength < [rawData length] && bytes[readLength] == 0x1e)
	{
//		SInt32  fieldLength = bytes[readLength + 3] + (bytes[readLength + 4] << 8) + (bytes[readLength + 5] << 16) + (bytes[readLength + 6] << 24);
		SInt32 *fieldLengthPointer = (SInt32 *)&bytes[readLength + 3];
		SInt32  fieldLength = NSSwapLittleLongToHost(*fieldLengthPointer);
		SInt32  fieldIndex  = bytes[readLength + 1] - 1;
		if (fieldIndex >= fieldNum)
		{
			return NO;
		}
		
		if (fieldLength == 0x01000000)
		{
			fieldLength = bytes[readLength + 2];
		}
		readLength += 7;
		
		dataFields[fieldIndex] = [NSData dataWithBytes:bytes + readLength length:fieldLength];
		
		readLength += fieldLength;
	}
	
	if (readLength == [rawData length])
	{
		return YES;
	}
	else if (readLength > [rawData length])
	{
		NSLog(@"ERROR. POINTER IS OVER THE DATA LENGTH");
		return NO;
	}
	else
	{
		NSLog(@"ERROR. Data field is not separated with correct code.");
		return NO;
	}
}

#pragma mark init and dealloc

- (id)init
{
	self = [super init];
	if (self != nil)
	{
		_fileDescriptor = -1;
		_status         = WXKPhoneStatusNotOpen;
	}
	return self;
}

- (void)dealloc
{
	[self closeConnection];
	
	[super dealloc];
}


#pragma mark Public methods

- (BOOL)openConnectionWithPassword:(NSString *)password progress:(NSObject <WXKProgressing> *)progress error:(NSError **)errorPtr
{
	_progress = [progress retain];
	[_progress setIndeterminate:YES];
	
	// -- open serial port --
	[_progress setMessage:NSLocalizedString(@"Opening Port...", @"phone.message.openPort")];
	
	int n = 0;
	while (true)
	{
		_fileDescriptor = open("/dev/cu.ah-k3001v.data", O_RDWR | O_NOCTTY | O_NONBLOCK);
		
		// When a port is successfully opened, exit current loop.
		if (_fileDescriptor != -1) break;
		
		// When a port cannot be opened, retry or return error.
		if (errno == ENOENT)
		{
			if (n++ < kOpeningRetryCount)
			{
				[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:kOpeningRetryInterval]]; 
				continue;
			}
			// @"The AIR-EDGE PHONE wasn't connected."
			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
														   code:WXKPhoneConnectionError
									   underlyingPOSIXErrorCode:errno
													   userInfo:nil];
			goto ERROR;
		}
		else
		{
			// @"Couldn't open a port for the AIR-EDGE PHONE."
			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
														   code:WXKPhoneOpenPortError
									   underlyingPOSIXErrorCode:errno
													   userInfo:nil];
			goto ERROR;
		}
	}
	// -- port opened --
	
	// -- configure port --
	[_progress setMessage:NSLocalizedString(@"Configuring Port...", @"phone.message.configurePort")];
	
	// set the serial port in exclusive use mode
	if (ioctl(_fileDescriptor, TIOCEXCL) == -1)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain 
													   code:WXKPhoneConfigurePortError
								   underlyingPOSIXErrorCode:errno
												   userInfo:nil];
		goto ERROR;
	}
	
	if (fcntl(_fileDescriptor, F_SETFL, 0) == -1)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain 
													   code:WXKPhoneConfigurePortError
								   underlyingPOSIXErrorCode:errno
												   userInfo:nil];
		goto ERROR;
	}
	
	
	// get and store the current attribute of the serial port
	if (tcgetattr(_fileDescriptor, &_originalTTYAttrs) == -1)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain 
													   code:WXKPhoneConfigurePortError
								   underlyingPOSIXErrorCode:errno
												   userInfo:nil];
		goto ERROR;
	}
	
	// change the attribute of the serial port
	struct termios options = _originalTTYAttrs;
	options.c_cflag = CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
	options.c_iflag = IGNBRK | IGNPAR;
	options.c_oflag = 0;
	options.c_lflag = 0;
	for (n = 0; n < NCCS; n++)
	{
		options.c_cc[n] = 0;
	}
	options.c_cc[VMIN]  = 1;
	options.c_cc[VTIME] = 0;
	
	if (tcsetattr(_fileDescriptor, TCSANOW, &options) == -1)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain 
													   code:WXKPhoneConfigurePortError
								   underlyingPOSIXErrorCode:errno
												   userInfo:nil];
		goto ERROR;
	}
	
	// -- port configured --
	_status = WXKPhoneStatusConfigured;
	
	// -- connecting --
	[_progress setMessage:NSLocalizedString(@"Connecting...", @"phone.message.password")];
	
	NSData *receivedData;
	n = 0;
	while (true)
	{
		if ([self sendCommand:0 subCommand:0x0101 parameter:"1" error:errorPtr] == NO) goto ERROR;
		if ((receivedData = [self receivedDataWithAddition:YES error:errorPtr]) == nil) goto ERROR;
		
		if (strncmp([receivedData bytes] + 12, "OK", 2))
		{
			if (n++ < kOpeningRetryCount)
			{
				[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:kOpeningRetryInterval]];
				continue;
			}
			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
														   code:WXKPhoneNotReadyError
													   userInfo:nil];
			goto ERROR;
		}
		break;
	}
	// -- connected --
	_status = WXKPhoneStatusConnected;
	
	// -- send password --
	[_progress setMessage:NSLocalizedString(@"Sending Security Code...", @"phone.message.password")];
	
	// convert password to C-string
	NSData *passwordData = [password dataUsingEncoding:NSShiftJISStringEncoding];
	if (passwordData == nil)
	{
		// in case security code contains characters which can be converted to SJIS encoding
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
													   code:WXKPhoneInapplicablePasswordStringEncodingError
												   userInfo:nil];
		goto ERROR;
	}
	int passwordDataLength = [passwordData length];
	if (passwordDataLength <= 0 || passwordDataLength > 4)
	{
		// in case password length is 0 or longer than 4
		if (errorPtr != NULL) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
															   code:WXKPhoneInapplicablePasswordStringLengthError
														   userInfo:nil];
		goto ERROR;
	}
	
	// send password to cellphone
	if ([self sendCommand:1 subCommand:0x0201 data:passwordData error:errorPtr] == NO) goto ERROR;
	if ((receivedData = [self receivedDataWithAddition:YES error:errorPtr]) == nil) goto ERROR;
	
	if(strncmp([receivedData bytes] + 12, "OK", 2))
	{
		if (errorPtr != NULL) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
															   code:WXKPhoneIncorrectPasswordError
														   userInfo:nil];
		goto ERROR;
	}
	
	_status = WXKPhoneStatusLoggedIn;
	// -- password sent --
	
	return YES;

ERROR:
	[self closeConnection];
	return NO;
}

- (void)closeConnection
{
	[_progress setIndeterminate:YES];
	[_progress setMessage:NSLocalizedString(@"Closing Connection...", @"phone.message.closePort")];
	
	if (_fileDescriptor != -1)
	{
		if (_status >= WXKPhoneStatusConnected)
		{
			[self sendCommand:0xff subCommand:0x0102 parameter:"" error:NULL];
		}
		
		if (_status >= WXKPhoneStatusConfigured)
		{
			tcdrain(_fileDescriptor);
			tcsetattr(_fileDescriptor, TCSANOW, &_originalTTYAttrs);
		}
		
		close(_fileDescriptor);
		_fileDescriptor = -1;
	}
	_status = WXKPhoneStatusNotOpen;
	[_progress release], _progress = nil;
}

- (NSArray *)arrayOfDataType:(WXKPhoneDataType)dataType error:(NSError **)errorPtr
{
	return [self receivedDataArrayOfType:dataType operation:WXKPhoneReceiveItemsOperation error:errorPtr];
}

- (BOOL)setArray:(NSArray *)dataArray ofDataType:(WXKPhoneDataType)dataType error:(NSError **)errorPtr
{
	// check number of items is under the limitation or not.
	const SInt32 count = [dataArray count];
	
	BOOL isInAppendMode = NO;
	
	NSData *receivedData;
	
	SInt32 n;
	
	// remove all items in cellphone
	if ([self sendCommand:3 subCommand:(dataType << 8) + 4 parameter:"13" error:errorPtr] == NO) goto ERROR;
	if ([self receivedDataWithAddition:YES error:errorPtr] == nil) goto ERROR;
	
	// begin to send all items
	if ([self sendCommand:3 subCommand:(dataType << 8) + 5 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ([self receivedDataWithAddition:YES error:errorPtr] == nil) goto ERROR;
	isInAppendMode = YES;
	
	[_progress setIndeterminate:NO];
	[_progress setHint:[NSString stringWithFormat:@"%d/%d", 0, count]];
	[_progress setMessage:NSLocalizedString(@"Sending...", @"phone.message.send")];
	
	for (n = 0; n < count; n++)
	{
		if ([_progress isRequestedToAbort])
		{
			if (errorPtr)
			{
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Writing is interrupted by user. The database of your cellular phone is not incomplete. Remind that if you read data from your cellular phone the database in your computer might be lost.", @"error.userCancelledWhileWriting.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:WXKErrorDomain
												code:WXKPhoneUserCancelledError
											userInfo:userInfo];
			}
			goto ERROR;
		}
		
		NSData *data = [dataArray objectAtIndex:n];
		
		// send a item.
		if ([self sendCommand:3 subCommand:(dataType << 8) + 3 data:data error:errorPtr] == NO) goto ERROR;
		if ((receivedData = [self receivedDataWithAddition:YES error:errorPtr]) == nil) goto ERROR;
		if(strncmp((const char *)[receivedData bytes] + 12, "0", 1))
		{
			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
														   code:WXKPhoneMalformedItemSentError
													   userInfo:nil];
			goto ERROR;
		}
		// refresh progress status
		[_progress incrementProgressBarBy:(1.0 / count)];
		[_progress setHint:[NSString stringWithFormat:@"%d/%d", n + 1, count]];
	}
	
	// finish sending items
	if ([self sendCommand:3 subCommand:(dataType << 8) + 6 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ([self receivedDataWithAddition:YES error:errorPtr] == nil) goto ERROR;
	
	return YES;
	
ERROR:
	if (isInAppendMode)
	{
		// stop receiving items
		[self sendCommand:3 subCommand:(dataType << 8) + 6 parameter:"" error:NULL];
		[self receivedDataWithAddition:YES error:NULL];
	}
	[self closeConnection];
	return NO;
}

- (NSArray *)listOfDataType:(WXKPhoneDataType)dataType error:(NSError **)errorPtr
{
	return [self receivedDataArrayOfType:dataType operation:WXKPhoneReceiveDirectoryOperation error:errorPtr];	  
}

- (NSArray *)dataArrayFromFiles:(NSArray *)files error:(NSError **)errorPtr
{
//	NSData *receivedData;
//	const UInt8 *bytes;
//	
//	NSAutoreleasePool *pool = nil;
//	NSMutableArray *mutableDataArray;
//	
//	NSEnumerator *enumerator = [files objectEnumerator];
//	NSManagedObject *file;
//	
//	mutableDataArray = [NSMutableArray arrayWithCapacity:[files count]];
//	
//	[_progress setIndeterminate:NO];
//	[_progress setMessage:NSLocalizedString(@"Receiving...", @"phone.message.readItems")];
//	
//	while (file = [enumerator nextObject])
//	{
//		pool = [[NSAutoreleasePool alloc] init];
//		
//		// Now safe to abort communication, so check whether user requested to abort progress
//		if ([_progress isRequestedToAbort])
//		{
//			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
//														   code:WXKPhoneUserCancelledError
//													   userInfo:nil];
//			goto ERROR;
//		}
//		
//		NSString *filename            = [file valueForKey:@"name"];
//		NSString *type                = [file valueForKey:@"type"];
//		NSString *filenameAndType     = [NSString stringWithFormat:@"%@\t%@\t", filename, type];
//		NSData   *filenameAndTypeData = [filenameAndType dataUsingEncoding:NSShiftJISStringEncoding];
//		if (filenameAndTypeData == nil)
//		{
//			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
//														   code:WXKPhoneFilenameAndTypeConversionError
//													   userInfo:nil];
//			return nil;
//		}
//		
//		if ([self sendCommand:3 subCommand:(WXKPhoneFileDataType << 8) + 2 data:filenameAndTypeData error:errorPtr] == NO) goto ERROR;
//		
//		// Read an item.
//		if ((receivedData = [self receivedDataWithAddition:YES error:errorPtr]) == nil) goto ERROR;
//		bytes = [receivedData bytes];
//		
//		[mutableDataArray addObject:receivedData];
//		
//		// refresh progress indicator
//		[_progress incrementProgressBarBy:(1.0 / [files count])];
//		
//		[pool release], pool = nil;
//	}
//	
////			if (bytes[3] != 0x03 || bytes[9] != dataType || bytes[11] != operationType)
////			{
////				if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
////															   code:WXKPhoneInapplicableDataReceivedError
////														   userInfo:nil];
////				goto ERROR;
////			}
////			
////			// Notify that reading has been finished
////			if ([self sendCommand:0x10 subCommand:0x0910 parameter:"0" error:errorPtr] == NO) goto ERROR;
//	
//	return mutableDataArray;
//	
//ERROR:
//	// Do not release pool! If you release this autorelease pool, the error 
//	// object is also released before a receiver of this method handles the error.
//	if (pool)
//	{
//		if (errorPtr) [*errorPtr retain];
//		[pool release], pool = nil;
//		if (errorPtr) [*errorPtr autorelease];
//	}
//	[self closeConnection];
	return nil;
}

#pragma mark Private high-level methods

- (NSArray *)receivedDataArrayOfType:(WXKPhoneDataType)dataType operation:(WXKPhoneOperation)operation error:(NSError **)errorPtr
{
	NSData *receivedData;
	const char *bytes;
	
	NSAutoreleasePool *pool = nil;
	NSMutableArray *mutableDataArray;
	
	// count number of items
	if ([self sendCommand:2 subCommand:(dataType << 8) + 1 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ((receivedData = [self receivedDataWithAddition:YES error:errorPtr]) == nil) goto ERROR;
	bytes = [receivedData bytes];
	const int count = atoi(bytes + 12);
	
	if (count > 0)
	{
		mutableDataArray = [NSMutableArray arrayWithCapacity:count];
		[_progress setIndeterminate:NO];
		[_progress setHint:[NSString stringWithFormat:@"%d/%d", 0, count]];
		[_progress setMessage:NSLocalizedString(@"Receiving...", @"phone.message.readItems")];
		
		// begin to read items.
		if ([self sendCommand:3 subCommand:(dataType << 8) + operation parameter:"" error:errorPtr] == NO) goto ERROR;
		
		int n;
		for (n = 0; n < count; n++)
		{
			pool = [[NSAutoreleasePool alloc] init];
			
			// Now safe to abort communication, so check whether user requested to abort progress
			if ([_progress isRequestedToAbort])
			{
				if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
															   code:WXKPhoneUserCancelledError
														   userInfo:nil];
				goto ERROR;
			}
			
			// Read an item.
			if ((receivedData = [self receivedDataWithAddition:YES error:errorPtr]) == nil) goto ERROR;
			bytes = [receivedData bytes];
			
			if (bytes[3] != 0x03 || bytes[9] != dataType || bytes[11] != operation)
			{
				if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
															   code:WXKPhoneInapplicableDataReceivedError
														   userInfo:nil];
				goto ERROR;
			}
			
			// Notify that reading has been finished
			if ([self sendCommand:0x10 subCommand:0x0910 parameter:"0" error:errorPtr] == NO) goto ERROR;
			
//			SInt32 dataLength = sizeof(buffer) > restBytes ? restBytes : sizeof(buffer);
//			[mutableDataArray addObject:[NSData dataWithBytes:buffer length:dataLength]];
			[mutableDataArray addObject:receivedData];
			
			// refresh progress indicator
			[_progress incrementProgressBarBy:(1.0 / count)];
			[_progress setHint:[NSString stringWithFormat:@"%d/%d", n+1, count]];
			
			[pool drain], pool = nil;
		}
	}
	return mutableDataArray;
	
ERROR:
	// Do not release pool! If you release this autorelease pool, the error 
	// object is also released before a receiver of this method handles the error.
	if (pool)
	{
		if (errorPtr) [*errorPtr retain];
		[pool drain], pool = nil;
		if (errorPtr) [*errorPtr autorelease];
	}
	[self closeConnection];
	return nil;
}

#pragma mark Mid-level read and write methods

- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(char *)parameter error:(NSError **)errorPtr
{
	NSData *data = [NSData dataWithBytesNoCopy:parameter length:strlen(parameter) freeWhenDone:NO];
	return [self sendCommand:a subCommand:b data:data error:errorPtr];
}

- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b data:(NSData *)data error:(NSError **)errorPtr
{
	UInt8 buf[12] = {
		0xe1, 0x01, (UInt8)(a >> 8), (UInt8)a,
		0, 0, 0, 0,
		0, (UInt8)(b >> 8), 0, (UInt8)b
	};
	
	const SInt32 lenWithHeader = [data length] + 4;
	buf[4] = (UInt8) lenWithHeader;
	buf[5] = (UInt8)(lenWithHeader >> 8 );
	buf[6] = (UInt8)(lenWithHeader >> 16);
	buf[7] = (UInt8)(lenWithHeader >> 24);
	
	if ([self sendRawBytes:buf length:sizeof(buf) error:errorPtr] == NO) return NO;
	
	if(data != nil)
	{
		if ([self sendRawBytes:[data bytes] length:[data length] error:errorPtr] == NO) return NO;
	}
	return YES;
}

- (NSData *)receivedDataWithAddition:(BOOL)hasAdditionalData error:(NSError **)errorPtr
{
	UInt8 buffer[kReadBufferSize];
	
	// write end mark on buffer
	buffer[12] = 0xff;
	buffer[13] = 0x00;
	
	int readLength = 0;
	while (readLength < 12)
	{
		// receive data
		SInt32 numBytes = [self receiveRawBytes:(buffer + readLength) maxLength:(12 - readLength) error:errorPtr];
		if (numBytes == -1) return NO;
		
		readLength += numBytes;
		
		// seek header mark
		int n;
		for (n = 0; n < readLength - 1; n++)
		{
			if (buffer[n] == 0xe1 && buffer[n + 1] == 0x02) break;
		}
		
		// if header mark is found, move buffer data so that header mark becomes the beginning of buffer
		if (n > 0)
		{
			readLength -= n;
			memmove(buffer, buffer + n, readLength);
		}
	}
	
	if (hasAdditionalData == NO)
	{
		return [NSData dataWithBytes:buffer length:12];
	}
	else
	{
		const int dataLength = buffer[4] + (((SInt32)buffer[5]) << 8) + (((SInt32)buffer[6]) << 16) + (((SInt32)buffer[7]) << 24) + 8;
		if (dataLength < 12)
		{
			if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
														   code:WXKPhoneInapplicableDataLengthReceivedError
													   userInfo:nil];
			return nil;
		}
		NSMutableData *mutableData = [NSMutableData dataWithCapacity:dataLength];
		[mutableData appendBytes:buffer length:12];
		
		int currentReadLength = ((dataLength - [mutableData length]) > sizeof(buffer)) ? sizeof(buffer) : (dataLength - [mutableData length]);
		while ([mutableData length] < dataLength)
		{
			SInt32 numBytes = [self receiveRawBytes:buffer maxLength:currentReadLength error:errorPtr];
			if (numBytes == -1) return nil;
			[mutableData appendBytes:buffer length:numBytes];
		}
		
		return mutableData;
	}
}

#pragma mark Lowest-level read and write methods

- (BOOL)sendRawBytes:(const UInt8 *)buffer length:(SInt32)length error:(NSError **)errorPtr
{
	SInt32 numBytes = write(_fileDescriptor, buffer, length);
	if (numBytes == -1)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
													   code:WXKPhoneWriteSerialPortError
								   underlyingPOSIXErrorCode:errno
												   userInfo:nil];
		return NO;
		
	}
	else if (numBytes != length)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
													   code:WXKPhoneWriteSerialPortError
												   userInfo:nil];
		return NO;
	}
	return YES;
}

- (SInt32)receiveRawBytes:(UInt8 *)buffer maxLength:(SInt32)bufferSize error:(NSError **)errorPtr
{
	SInt32 numBytes = read(_fileDescriptor, buffer, bufferSize);
	
	if (numBytes == -1)
	{
		if (errorPtr) *errorPtr = [WXKError errorWithDomain:WXKErrorDomain
													   code:WXKPhoneReadSerialPortError
								   underlyingPOSIXErrorCode:errno
												   userInfo:nil];
		return -1;
	}
	return numBytes;
}

@end